//
//  TodoList.swift
//  Do It
//
//  Created by Jim Dovey on 8/25/19.
//  Copyright © 2019 Jim Dovey. All rights reserved.
//

import SwiftUI
import CoreData
import Combine

struct TodoList: View {
    // START:NewListData
    private enum ListData {
        // START_HIGHLIGHT
        case list(TodoItemList)
        case items(LocalizedStringKey, NSFetchRequest<TodoItem>)
        // END_HIGHLIGHT
        case group(TodoItemGroup)
    }
    // END:NewListData
    
    private enum Sheet: Identifiable, Hashable {
        case itemEditor
        case listEditor
        
        var id: Sheet { self }
    }
    
    @State private var presentedSheet: Sheet? = nil

    @State private var sortBy: SortOption = .manual
    @State private var showingChooser: Bool = false
    @Environment(\.presentationMode) private var presentationMode
    // START:NewListData

    @Environment(\.managedObjectContext) var objectContext

    @State private var listData: ListData
    @MutableFetchRequest<TodoItem> var items: MutableFetchedResults<TodoItem> //<label id="code.7.todolist.fetchrequest.mutable"/>
    // END:NewListData

    // START:NewInitializers
    init(list: TodoItemList) {
        self._listData = State(wrappedValue: .list(list))
        let request = list.requestForAllItems
        request.sortDescriptors = SortOption.manual.sortDescriptors
        self._items = MutableFetchRequest(fetchRequest: request)
    }

    init(title: LocalizedStringKey, fetchRequest: NSFetchRequest<TodoItem>) {
        let request = fetchRequest.copy() as! NSFetchRequest<TodoItem>
        request.sortDescriptors = SortOption.manual.sortDescriptors
        self._listData = State(wrappedValue: .items(title, request))
        self._items = MutableFetchRequest(fetchRequest: request)
    }

    init(group: TodoItemGroup) {
        self._listData = State(wrappedValue: .group(group))
        let request = group.fetchRequest//<label id="code.7.list.init.group.request"/>
        request.sortDescriptors = SortOption.manual.sortDescriptors
        self._items = MutableFetchRequest(fetchRequest: request)
    }
    // END:NewInitializers

    var body: some View {
        List {
            // START:UpdateForEach
            // START_HIGHLIGHT
            ForEach(items) { item in
                // END_HIGHLIGHT
                NavigationLink(destination: TodoItemDetail(item: item)) {
                    TodoItemRow(item: item)
                        .accentColor(self.color(for: item))
                }
            }
            // END:UpdateForEach
            // START:OnChange
            .onDelete(perform: self.removeTodoItems(atOffsets:))
            .onMove(perform: self.sortBy == .manual
                ? self.moveTodoItems(fromOffsets:to:)
                : nil)
            // END:OnChange
        }
        .navigationBarTitle(title)
        .navigationBarItems(trailing: barItems)
        .listStyle(GroupedListStyle())
        .actionSheet(isPresented: $showingChooser) {
            ActionSheet(
                title: Text("Sort Order"),
                buttons: SortOption.allCases.map { opt in
                    ActionSheet.Button.default(Text(opt.title)) {
                        self.updateSortDescriptors(opt)
                    }
            })
        }
        .sheet(
            item: $presentedSheet,
            onDismiss: { try? self.objectContext.save() },
            content: presentEditor(of:))
    }
}

// MARK: - Helper Properties

extension TodoList {
    private var sortButton: some View {
        Button(action: { self.showingChooser.toggle() }) {
            Image(systemName: "arrow.up.arrow.down.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Sort List"))
        }
    }

    private var addButton: some View {
        // START:AddButtonAction
        Button(action: {
            // START_HIGHLIGHT
            self.presentedSheet = .itemEditor
            // END_HIGHLIGHT
        }) {
            // <literal:elide> ... </literal:elide>
            // END:AddButtonAction
            Image(systemName: "plus.circle.fill")
                .imageScale(.large)
                .accessibility(label: Text("Add New To-Do Item"))
            // START:AddButtonAction
        }
        // END:AddButtonAction
    }

    // START:NewEditorSheet
    private var editorSheet: some View {
        let editContext = objectContext.editingContext()
        let editList: TodoItemList
        if let list = self.list {
            editList = editContext.realize(list)! //<label id="code.7.itemlist.editor.item.list"/>
        }
        else {
            editList = TodoItemList.defaultList(in: editContext)
        }
        let editingItem = TodoItem.newTodoItem(in: editList)
        
        return NavigationView {
            TodoItemEditor(item: editingItem)
                .environment(\.managedObjectContext, editContext)
        }
    }
    // END:NewEditorSheet

    private var barItems: some View {
        HStack(spacing: 14) {
            if isList {
                Button(action: { self.presentedSheet = .listEditor }) {
                    Image(systemName: "info.circle.fill")
                        .imageScale(.large)
                        .accessibility(label: Text("Show List Information"))
                }
            }
            sortButton
            addButton
            EditButton()
        }
    }
    
    private func presentEditor(of type: Sheet) -> some View {
        switch type {
        case .itemEditor:
            return AnyView(self.editorSheet)
            // START:PresentListEditor
        case .listEditor:
            guard let list = self.list else {
                preconditionFailure("List editor presented with no list!")
            }
            let context = self.objectContext.editingContext()
            guard let editList = context.realize(list) else {
                preconditionFailure("List editor: can't create editing list!")
            }
            return AnyView(TodoListEditor(list: editList)
                .environment(\.managedObjectContext, context))
            // END:PresentListEditor
        }
    }

    private var isList: Bool {
        if case .list = self.listData {
            return true
        }
        return false
    }
    
    private var list: TodoItemList? {
        if case .list(let list) = self.listData {
            return list
        }
        return nil
    }

    private func forciblyDismiss() {
        presentationMode.wrappedValue.dismiss()
    }

    // START:TitleColor
    private var title: LocalizedStringKey {
        switch listData {
            // START_HIGHLIGHT
        case .list(let list): return LocalizedStringKey(list.name!)
            // END_HIGHLIGHT
        case .items(let name, _): return name
        case .group(let group): return group.title
        }
    }

    // START_HIGHLIGHT
    private func color(for item: TodoItem) -> Color {
        // END_HIGHLIGHT
        switch listData {
        case .list(let list): return list.color.uiColor
            // START_HIGHLIGHT
        case .items: return item.list!.color.uiColor
            // END_HIGHLIGHT
        case .group(let group): return group.color
        }
    }
    // END:TitleColor
}

// MARK: - Sorting

extension TodoList {
    // START:NewSorting
    private func updateSortDescriptors(_ sortOption: SortOption) {
        let request = self._items.fetchRequest//<label id="code.7.todolist.sort.getrequest"/>
        if case .group(.all) = self.listData, case .manual = sortOption {
            let listOrder = NSSortDescriptor(key: "list.manualSortOrder",
                                             ascending: true) //<label id="code.7.todolist.sort.listorder"/>
            request.sortDescriptors = [listOrder] + sortOption.sortDescriptors
        }
        else {
            request.sortDescriptors = sortOption.sortDescriptors
        }
        self._items.fetchRequest = request//<label id="code.7.todolist.fetchrequest.update"/>
    }
    // END:NewSorting
}

// MARK: - Model Manipulation

extension TodoList {
    // START:RemoveItems
    private func removeTodoItems(atOffsets offsets: IndexSet) {
        for index in offsets {
            self.objectContext.delete(self.items[index])
        }
        UIApplication.shared.saveContext()
    }
    // END:RemoveItems

    // START:MoveItems
    private func moveTodoItems(fromOffsets offsets: IndexSet, to newIndex: Int) {
        var newOrder = Array(items)
        newOrder.move(fromOffsets: offsets, toOffset: newIndex)
        for (index, item) in newOrder.enumerated() {
            item.manualSortOrder = Int32(index)
        }
        UIApplication.shared.saveContext()
    }
    // END:MoveItems
}

fileprivate enum SortOption: String, CaseIterable {
    case title = "Title"
    case priority = "Priority"
    case dueDate = "Due Date"
    case manual = "Manual"

    var title: LocalizedStringKey { LocalizedStringKey(rawValue) }

    // START:SortDescriptors
    var sortDescriptors: [NSSortDescriptor] {
        switch self {
        case .title:
            return [NSSortDescriptor(keyPath: \TodoItem.sortingTitle,
                                     ascending: true)]
        case .priority:
            return [NSSortDescriptor(keyPath: \TodoItem.rawPriority,
                                     ascending: false)]
        case .dueDate:
            return [NSSortDescriptor(keyPath: \TodoItem.date,
                                     ascending: true)]
        case .manual:
            return [NSSortDescriptor(keyPath: \TodoItem.manualSortOrder,
                                     ascending: true)]
        }
    }
    // END:SortDescriptors
}

struct TodoList_Previews: PreviewProvider {
    static var previews: some View {
        // START:Preview
        NavigationView {
            TodoList(group: .all)
                .environment(\.managedObjectContext,
                             PreviewDataStore.shared.viewContext)
        }
        // END:Preview
    }
}
